(function($, PLUGIN_ID) {
  'use strict';

  // 設定キー定義
  const CONF = {
    MODE: 'download_mode', // individual, bulk, both

    // 個別DL用設定
    INDIV_TARGET_TYPE: 'indiv_target_type', // fixed, runtime
    INDIV_TARGET_FIELDS: 'indiv_target_fields', // JSON string array
    INDIV_NAME_TYPE: 'indiv_name_type', // default, custom
    INDIV_NAME_FIELD: 'indiv_name_field', // field code
    INDIV_NAME_ADD_DATE: 'indiv_name_add_date', // true/false
    INDIV_FOLDER_TYPE: 'indiv_folder_type', // split, flat

    // 一括DL用設定
    BULK_TARGET_TYPE: 'bulk_target_type', // fixed, runtime
    BULK_TARGET_FIELDS: 'bulk_target_fields', // JSON string array
    BULK_NAME_TYPE: 'bulk_name_type', // default, custom
    BULK_NAME_TEXT: 'bulk_name_text', // string
    BULK_NAME_ADD_DATE: 'bulk_name_add_date', // true/false
    BULK_FOLDER_TYPE: 'bulk_folder_type', // split, flat
    BULK_FOLDER_FIELD: 'bulk_folder_field' // field code
  };

  // エスケープ関数
  function escapeHtml(str) {
    if (typeof str !== 'string') return str;
    return str.replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#39;');
  }

  // チェックボックスリスト生成関数
  function createCheckboxList(fields, containerId, selectedValues = []) {
    const $container = $('#' + containerId);
    $container.empty();

    if (fields.length === 0) {
      $container.append('<div>添付ファイルフィールドが見つかりません。</div>');
      return;
    }

    fields.forEach(field => {
      const isChecked = selectedValues.includes(field.code) ? 'checked' : '';
      const html = `
        <div class="kintoneplugin-input-checkbox-item">
          <input type="checkbox" name="${containerId}-chk" value="${escapeHtml(field.code)}" id="${containerId}-${escapeHtml(field.code)}" ${isChecked}>
          <label for="${containerId}-${escapeHtml(field.code)}">${escapeHtml(field.label)} (${escapeHtml(field.code)})</label>
        </div>
      `;
      $container.append(html);
    });
  }

  // ドロップダウン生成関数
  function createDropdown($select, fields, selectedValue) {
    $select.empty();
    $select.append('<option value="">-- 指定なし --</option>');
    fields.forEach(field => {
      const isSelected = (field.code === selectedValue) ? 'selected' : '';
      $select.append(`<option value="${escapeHtml(field.code)}" ${isSelected}>${escapeHtml(field.label)} (${escapeHtml(field.code)})</option>`);
    });
  }

  // メイン処理
  $(document).ready(function() {
    // 1. Kintoneフィールド情報の取得
    kintone.api(kintone.api.url('/k/v1/preview/app/form/fields', true), 'GET', {app: kintone.app.getId()}, function(resp) {
      const properties = resp.properties;
      const attachmentFields = [];
      const singleLineFields = [];

      // フィールドの仕分け
      for (const code in properties) {
        const field = properties[code];
        if (field.type === 'FILE') {
          attachmentFields.push({code: field.code, label: field.label});
        } else if (field.type === 'SINGLE_LINE_TEXT') {
          singleLineFields.push({code: field.code, label: field.label});
        }
      }

      // 設定値の読み込み
      const config = kintone.plugin.app.getConfig(PLUGIN_ID);
      const savedIndivFields = config[CONF.INDIV_TARGET_FIELDS] ? JSON.parse(config[CONF.INDIV_TARGET_FIELDS]) : [];
      const savedBulkFields = config[CONF.BULK_TARGET_FIELDS] ? JSON.parse(config[CONF.BULK_TARGET_FIELDS]) : [];

      // UIの生成
      createCheckboxList(attachmentFields, 'indiv-attachment-list', savedIndivFields);
      createCheckboxList(attachmentFields, 'bulk-attachment-list', savedBulkFields);
      createDropdown($('#indiv-name-field'), singleLineFields, config[CONF.INDIV_NAME_FIELD]);
      createDropdown($('#bulk-folder-field'), singleLineFields, config[CONF.BULK_FOLDER_FIELD]);

      // 保存された値の反映 (ラジオボタン/チェックボックス/テキスト)
      if (config[CONF.MODE]) {
          $(`input[name="download-mode"][value="${config[CONF.MODE]}"]`).prop('checked', true);
      }

      // 個別DL設定反映
      if (config[CONF.INDIV_TARGET_TYPE]) {
          $(`input[name="indiv-target-type"][value="${config[CONF.INDIV_TARGET_TYPE]}"]`).prop('checked', true);
      }
      if (config[CONF.INDIV_NAME_TYPE]) {
          $(`input[name="indiv-name-type"][value="${config[CONF.INDIV_NAME_TYPE]}"]`).prop('checked', true);
      }
      $('#indiv-name-add-date').prop('checked', config[CONF.INDIV_NAME_ADD_DATE] === 'true');
      if (config[CONF.INDIV_FOLDER_TYPE]) {
          $(`input[name="indiv-folder-type"][value="${config[CONF.INDIV_FOLDER_TYPE]}"]`).prop('checked', true);
      }

      // 一括DL設定反映
      if (config[CONF.BULK_TARGET_TYPE]) {
          $(`input[name="bulk-target-type"][value="${config[CONF.BULK_TARGET_TYPE]}"]`).prop('checked', true);
      }
      if (config[CONF.BULK_NAME_TYPE]) {
          $(`input[name="bulk-name-type"][value="${config[CONF.BULK_NAME_TYPE]}"]`).prop('checked', true);
      }
      $('#bulk-custom-name-text').val(config[CONF.BULK_NAME_TEXT] || '');
      $('#bulk-name-add-date').prop('checked', config[CONF.BULK_NAME_ADD_DATE] === 'true');
      if (config[CONF.BULK_FOLDER_TYPE]) {
          $(`input[name="bulk-folder-type"][value="${config[CONF.BULK_FOLDER_TYPE]}"]`).prop('checked', true);
      }

      // 初期表示の切り替え実行
      updateVisibility();
    });

    // --- イベントハンドラ ---

    // モード切り替え
    $('input[name="download-mode"]').on('change', updateVisibility);
    
    // 個別DL: ターゲットタイプ切り替え
    $('input[name="indiv-target-type"]').on('change', function() {
        $('#indiv-fixed-container').toggle($(this).val() === 'fixed');
    });
    // 個別DL: ファイル名タイプ切り替え
    $('input[name="indiv-name-type"]').on('change', function() {
        $('#indiv-custom-name-container').toggle($(this).val() === 'custom');
    });

    // 一括DL: ターゲットタイプ切り替え
    $('input[name="bulk-target-type"]').on('change', function() {
        $('#bulk-fixed-container').toggle($(this).val() === 'fixed');
    });
    // 一括DL: ファイル名タイプ切り替え
    $('input[name="bulk-name-type"]').on('change', function() {
        $('#bulk-custom-name-container').toggle($(this).val() === 'custom');
    });
    // 一括DL: フォルダタイプ切り替え
    $('input[name="bulk-folder-type"]').on('change', function() {
        $('#bulk-split-settings').toggle($(this).val() === 'split');
    });

    // 表示制御関数
    function updateVisibility() {
      const mode = $('input[name="download-mode"]:checked').val();
      
      // セクションの表示
      if (mode === 'individual') {
          $('#section-individual').show();
          $('#section-bulk').hide();
      } else if (mode === 'bulk') {
          $('#section-individual').hide();
          $('#section-bulk').show();
      } else {
          $('#section-individual').show();
          $('#section-bulk').show();
      }

      // 内部要素の初期表示（トグルイベントを発火させて合わせる）
      $('input[name="indiv-target-type"]:checked').trigger('change');
      $('input[name="indiv-name-type"]:checked').trigger('change');
      $('input[name="bulk-target-type"]:checked').trigger('change');
      $('input[name="bulk-name-type"]:checked').trigger('change');
      $('input[name="bulk-folder-type"]:checked').trigger('change');
    }

    // 保存処理
    $('#btn-save').on('click', function() {
      const config = {};

      // 基本設定
      config[CONF.MODE] = $('input[name="download-mode"]:checked').val();

      // 個別DL設定取得
      config[CONF.INDIV_TARGET_TYPE] = $('input[name="indiv-target-type"]:checked').val();
      const indivSelectedFields = [];
      $('input[name="indiv-attachment-list-chk"]:checked').each(function() {
          indivSelectedFields.push($(this).val());
      });
      config[CONF.INDIV_TARGET_FIELDS] = JSON.stringify(indivSelectedFields);
      config[CONF.INDIV_NAME_TYPE] = $('input[name="indiv-name-type"]:checked').val();
      config[CONF.INDIV_NAME_FIELD] = $('#indiv-name-field').val();
      config[CONF.INDIV_NAME_ADD_DATE] = String($('#indiv-name-add-date').is(':checked'));
      config[CONF.INDIV_FOLDER_TYPE] = $('input[name="indiv-folder-type"]:checked').val();

      // 一括DL設定取得
      config[CONF.BULK_TARGET_TYPE] = $('input[name="bulk-target-type"]:checked').val();
      const bulkSelectedFields = [];
      $('input[name="bulk-attachment-list-chk"]:checked').each(function() {
          bulkSelectedFields.push($(this).val());
      });
      config[CONF.BULK_TARGET_FIELDS] = JSON.stringify(bulkSelectedFields);
      config[CONF.BULK_NAME_TYPE] = $('input[name="bulk-name-type"]:checked').val();
      config[CONF.BULK_NAME_TEXT] = $('#bulk-custom-name-text').val();
      config[CONF.BULK_NAME_ADD_DATE] = String($('#bulk-name-add-date').is(':checked'));
      config[CONF.BULK_FOLDER_TYPE] = $('input[name="bulk-folder-type"]:checked').val();
      config[CONF.BULK_FOLDER_FIELD] = $('#bulk-folder-field').val();

      kintone.plugin.app.setConfig(config, function() {
        alert('設定を保存しました。アプリを更新してください。');
        window.location.href = '../../flow?app=' + kintone.app.getId();
      });
    });

    // キャンセル
    $('#btn-cancel').on('click', function() {
      window.location.href = '../../flow?app=' + kintone.app.getId();
    });
  });
})(jQuery, kintone.$PLUGIN_ID);