// 即時関数で全体を囲む
(function($, PLUGIN_ID) {
  'use strict';

  // --- 設定値の読み込み ---
  const CONFIG = kintone.plugin.app.getConfig(PLUGIN_ID);
  if (!CONFIG) return;

  const CONF = {
    MODE: CONFIG.download_mode || 'both', 

    // 個別DL設定
    INDIV_TARGET_TYPE: CONFIG.indiv_target_type || 'fixed',
    INDIV_TARGET_FIELDS: CONFIG.indiv_target_fields ? JSON.parse(CONFIG.indiv_target_fields) : [],
    INDIV_NAME_TYPE: CONFIG.indiv_name_type || 'default',
    INDIV_NAME_FIELD: CONFIG.indiv_name_field,
    INDIV_NAME_ADD_DATE: CONFIG.indiv_name_add_date === 'true',
    INDIV_FOLDER_TYPE: CONFIG.indiv_folder_type || 'split',

    // 一括DL設定
    BULK_TARGET_TYPE: CONFIG.bulk_target_type || 'fixed',
    BULK_TARGET_FIELDS: CONFIG.bulk_target_fields ? JSON.parse(CONFIG.bulk_target_fields) : [],
    BULK_NAME_TYPE: CONFIG.bulk_name_type || 'default',
    BULK_NAME_TEXT: CONFIG.bulk_name_text || '',
    BULK_NAME_ADD_DATE: CONFIG.bulk_name_add_date === 'true',
    BULK_FOLDER_TYPE: CONFIG.bulk_folder_type || 'split',
    BULK_FOLDER_FIELD: CONFIG.bulk_folder_field
  };

  // --- 定数 ---
  const WARN_THRESHOLD_SIZE = 1 * 1024 * 1024 * 1024; // 1GB

  // --- ヘルパー関数 ---

  function getTodayDateString() {
    const d = new Date();
    return d.getFullYear() +
           ('0' + (d.getMonth() + 1)).slice(-2) +
           ('0' + d.getDate()).slice(-2);
  }

  function sanitizeFileName(str) {
    return (str || '').replace(/[\\/:*?"<>|]/g, '_').trim();
  }

  function escapeHtml(str) {
    if (typeof str !== 'string') return str;
    return str.replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#39;');
  }

  function formatBytes(bytes, decimals = 2) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const dm = decimals < 0 ? 0 : decimals;
    const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
  }

  // ★ライセンス認証関数 (サイレント実行)
  function checkPaidLicense(subdomain, onSuccess) {
    // 以前ここにあった「Swal.fire({ title: 'ライセンス確認中...' })」を削除しました。
    // これにより、ボタン押下時は何も表示されず、裏で通信だけ行われます。

    const proxyUrl = `https://en-noshita.co.jp/plugin_license/proxy.php?subdomain=${subdomain}`;
    kintone.proxy(proxyUrl, 'GET', {}, {}, function(body, status) {
      try {
        const response = JSON.parse(body);
        if (status === 200 && response.records.length > 0) {
          const record = response.records[0];
          const statusField = record['稼働状況'].value;
          
          if (statusField === '有効') {
            // 成功時は何も表示せず、次の処理へ進む
            onSuccess();
          } else {
            Swal.fire({
              icon: 'error',
              title: 'ライセンスが無効です',
              html: '<div style="text-align:left; font-size:14px;">いずれかの理由によりライセンスが無効です。ご確認ください。<br><br>' +
                    '1. 料金の支払いがされていない：発行されている請求書を確認の上、お支払いください。<br>' +
                    '2. 料金を支払ったが無効状態：料金のお支払いの翌営業日に有効化されていない場合、ご登録されているドメインが間違っている可能性があります。お手数ですがお問い合わせください。<br><br>' +
                    'お問合せ先：info@en-noshita.co.jp</div>'
            });
          }
        } else {
          Swal.fire({
            icon: 'error',
            title: 'ライセンス情報が見つかりません',
            html: '<div style="text-align:left; font-size:14px;">以下をご確認ください。<br><br>' +
                  '1. 料金を支払ったにも関わらず翌営業日になっても無効の場合、ご登録されているドメインが間違っている可能性があります。お手数ですがお問い合わせください。<br>' +
                  '2. 料金が未払の可能性があります。ご請求をご確認ください。<br><br>' +
                  'お問合せ先：info@en-noshita.co.jp</div>'
          });
        }
      } catch (e) {
        console.error('レスポンス解析エラー:', e);
        Swal.fire('エラー', 'ライセンス情報の取得に失敗しました。', 'error');
      }
    }, function(error) {
      console.error('ライセンスチェック中にエラーが発生しました:', error);
      Swal.fire('通信エラー', '通信エラーが発生しました。再度お試しください。', 'error');
    });
  }

  // アプリの添付ファイルフィールド一覧を取得 (本番API)
  async function getAttachmentFields() {
    try {
      const resp = await kintone.api(kintone.api.url('/k/v1/app/form/fields', true), 'GET', {app: kintone.app.getId()});
      const fields = [];
      for (const code in resp.properties) {
        if (resp.properties[code].type === 'FILE') {
          fields.push({ code: code, label: resp.properties[code].label });
        }
      }
      return fields;
    } catch (e) {
      console.error('フィールド情報の取得に失敗しました', e);
      return [];
    }
  }

  async function downloadFileBlob(fileKey) {
    const url = kintone.api.url('/k/v1/file', true) + '?fileKey=' + fileKey;
    const response = await fetch(url, {
      headers: { 'X-Requested-With': 'XMLHttpRequest' }
    });
    if (!response.ok) throw new Error('File download failed');
    return await response.blob();
  }

  // --- UI: SweetAlert2 ---

  // フィールド選択モーダル
  async function showFieldSelectModal(fields) {
    if (!fields || fields.length === 0) {
        Swal.fire('情報', 'ダウンロード可能な添付ファイルフィールドが見つかりません。', 'info');
        return null;
    }

    const style = `
      <style>
        .swal-field-list-container {
          text-align: left;
          max-height: 400px;
          overflow-y: auto;
          padding: 10px;
          border: 1px solid #e3e7e8;
          border-radius: 4px;
          background-color: #f7f9fa;
        }
        .swal-field-item {
          display: flex;
          align-items: center;
          background-color: #fff;
          border: 1px solid #dcdcdc;
          border-radius: 4px;
          padding: 12px 16px;
          margin-bottom: 8px;
          cursor: pointer;
          transition: all 0.2s ease;
          position: relative;
        }
        .swal-field-item:hover {
          border-color: #3498db;
          background-color: #eaf6fd;
          transform: translateY(-1px);
          box-shadow: 0 2px 5px rgba(0,0,0,0.08);
        }
        .swal-field-chk {
          transform: scale(1.3);
          margin-right: 16px;
          cursor: pointer;
        }
        .swal-field-info {
          display: flex;
          flex-direction: column;
          flex-grow: 1;
        }
        .swal-field-label {
          font-size: 15px;
          font-weight: bold;
          color: #333;
          line-height: 1.4;
        }
        .swal-field-code {
          font-size: 12px;
          color: #888;
          margin-top: 2px;
          font-family: monospace;
        }
        .swal-field-item:has(.swal-field-chk:checked) {
          border-color: #3498db;
          background-color: #f0f9ff;
        }
      </style>
    `;

    let htmlContent = `<div class="swal-field-list-container">`;
    fields.forEach((field, index) => {
        const uniqueId = `swal-chk-${index}`;
        htmlContent += `
            <label class="swal-field-item" for="${uniqueId}">
                <input type="checkbox" class="swal-field-chk" value="${escapeHtml(field.code)}" id="${uniqueId}">
                <div class="swal-field-info">
                    <span class="swal-field-label">${escapeHtml(field.label)}</span>
                    <span class="swal-field-code">${escapeHtml(field.code)}</span>
                </div>
            </label>`;
    });
    htmlContent += '</div>';

    const { value: selectedFields } = await Swal.fire({
      title: 'ダウンロードするフィールドを選択',
      html: style + htmlContent, 
      showCancelButton: true,
      confirmButtonText: '<i class="fa fa-download"></i> ダウンロード開始',
      cancelButtonText: 'キャンセル',
      width: '600px',
      focusConfirm: false,
      preConfirm: () => {
        const checked = document.querySelectorAll('.swal-field-chk:checked');
        if (checked.length === 0) {
            Swal.showValidationMessage('少なくとも1つ選択してください');
            return false;
        }
        return Array.from(checked).map(input => input.value);
      }
    });
    return selectedFields;
  }

  function showProgressModal(title) {
    Swal.fire({
      title: title,
      html: '<div style="margin: 10px 0;">準備中...</div><div class="swal2-progress-bar" style="display:flex; height: 10px; background: #eee; width: 100%;"><div id="swal-progress-inner" style="width: 0%; background: #3498db; transition: width 0.2s;"></div></div>',
      allowOutsideClick: false,
      allowEscapeKey: false,
      showConfirmButton: false,
      didOpen: () => { Swal.showLoading(); }
    });
  }

  function updateProgress(current, total, message) {
    const percentage = Math.floor((current / total) * 100);
    const content = Swal.getHtmlContainer();
    if (content) {
      const msgDiv = content.querySelector('div:first-child');
      const barDiv = content.querySelector('#swal-progress-inner');
      if (msgDiv) msgDiv.textContent = `${message} (${current}/${total})`;
      if (barDiv) barDiv.style.width = `${percentage}%`;
    }
  }

  // --- コアロジック ---

  async function processDownload(records, targetFieldCodes, zipFileName, options) {
    if (!targetFieldCodes || targetFieldCodes.length === 0) return;

    let totalFiles = 0;
    let totalSize = 0; 
    const downloadTasks = [];
    
    // フォルダ名の重複管理用
    const folderNameCounts = {}; 
    const recordIdToFolderName = {};
    const usedFilePaths = new Set();

    // 1. 全レコードを走査してフォルダ名確定 & サイズ計算
    records.forEach(record => {
      const recordId = record['$id'].value;
      let baseFolderName = recordId;

      if (options.isBulk && options.folderType === 'split' && options.folderField && record[options.folderField]) {
        const val = record[options.folderField].value;
        if (val) {
          baseFolderName = sanitizeFileName(val);
        }
      }

      if (folderNameCounts[baseFolderName] === undefined) {
        folderNameCounts[baseFolderName] = 0;
        recordIdToFolderName[recordId] = baseFolderName;
      } else {
        folderNameCounts[baseFolderName]++;
        const count = folderNameCounts[baseFolderName];
        recordIdToFolderName[recordId] = `${baseFolderName} (${count})`;
      }

      targetFieldCodes.forEach(fieldCode => {
        if (record[fieldCode] && record[fieldCode].value && record[fieldCode].value.length > 0) {
          record[fieldCode].value.forEach(file => {
            totalFiles++;
            
            const fileSize = parseInt(file.size, 10) || 0;
            totalSize += fileSize;

            downloadTasks.push({
              record: record,
              fieldCode: fieldCode,
              file: file
            });
          });
        }
      });
    });

    if (totalFiles === 0) {
      Swal.fire('情報', '対象の添付ファイルがありませんでした。', 'info');
      return;
    }

    // ★容量警告チェック (一括DL かつ 1GB超えの場合のみ)
    if (options.isBulk && totalSize > WARN_THRESHOLD_SIZE) {
        const result = await Swal.fire({
            title: '大容量ファイルの確認',
            html: `合計ファイルサイズ: <b>${formatBytes(totalSize)}</b> を処理しようとしています。<br><br>
                   お使いのデバイススペックによってはメモリ不足により処理がフリーズすることがございます。<br>
                   うまくいかない場合は対象レコードを絞り込み、件数を減らして複数回に分けてお試しください。`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: '了承して続行',
            cancelButtonText: 'やめる',
            confirmButtonColor: '#d33',
            width: '600px'
        });

        if (!result.isConfirmed) {
            return; // キャンセルされたら終了
        }
    }


    showProgressModal('ファイルをダウンロード中...');
    const zip = new JSZip();
    let processedCount = 0;

    const rootFolderName = zipFileName.replace(/\.zip$/i, '');

    const CHUNK_SIZE = 5; 
    for (let i = 0; i < downloadTasks.length; i += CHUNK_SIZE) {
      const chunk = downloadTasks.slice(i, i + CHUNK_SIZE);
      
      await Promise.all(chunk.map(async (task) => {
        try {
          const blob = await downloadFileBlob(task.file.fileKey);
          
          let folderPath = '';
          const fileName = task.file.name;

          if (options.isBulk) {
            // 一括DL
            if (options.folderType === 'split') {
              const recordId = task.record['$id'].value;
              const recordDirName = recordIdToFolderName[recordId];
              const fieldDirName = sanitizeFileName(task.fieldCode); 
              folderPath = `${rootFolderName}/${recordDirName}/${fieldDirName}/`;
            } else {
              folderPath = `${rootFolderName}/`;
            }
          } else {
            // 個別DL
            if (options.folderType === 'split') {
              const fieldDirName = sanitizeFileName(task.fieldCode);
              folderPath = `${rootFolderName}/${fieldDirName}/`;
            } else {
              folderPath = `${rootFolderName}/`;
            }
          }
          
          // ファイル名の重複チェックと連番付与
          let fullPath = folderPath + fileName;
          
          if (usedFilePaths.has(fullPath)) {
            const dotIndex = fileName.lastIndexOf('.');
            let nameBase = fileName;
            let extension = '';
            if (dotIndex !== -1) {
                nameBase = fileName.substring(0, dotIndex);
                extension = fileName.substring(dotIndex);
            }
            
            let counter = 1;
            while (usedFilePaths.has(fullPath)) {
                fullPath = `${folderPath}${nameBase} (${counter})${extension}`;
                counter++;
            }
          }
          
          usedFilePaths.add(fullPath);
          zip.file(fullPath, blob);

        } catch (e) {
          console.error('Download failed:', e);
        } finally {
          processedCount++;
          updateProgress(processedCount, totalFiles, 'ダウンロード処理中...');
        }
      }));
    }

    updateProgress(totalFiles, totalFiles, 'ZIP圧縮中...');
    const zipBlob = await zip.generateAsync({ type: 'blob' });
    saveAs(zipBlob, zipFileName);

    Swal.fire({
      icon: 'success',
      title: '完了',
      text: 'ダウンロードが完了しました。',
    });
  }


  // --- メイン処理 ---
  async function handleBulkDownload() {
    const subdomain = location.host.split('.')[0];
    // ★ライセンス認証: 成功時のみ実行
    checkPaidLicense(subdomain, async function() {
      
      let targetFields = [];
      if (CONF.BULK_TARGET_TYPE === 'fixed') {
        targetFields = CONF.BULK_TARGET_FIELDS;
      } else {
        const allFields = await getAttachmentFields();
        targetFields = await showFieldSelectModal(allFields);
        if (!targetFields) return;
      }

      if (targetFields.length === 0) {
        Swal.fire('エラー', '対象フィールドが設定されていません。', 'error');
        return;
      }

      const query = kintone.app.getQueryCondition() || '';
      showProgressModal('レコード情報を取得中...');
      
      const fetchAll = async () => {
        let records = [];
        let offset = 0;
        const limit = 500;
        const fields = ['$id', ...targetFields];
        if (CONF.BULK_FOLDER_FIELD) fields.push(CONF.BULK_FOLDER_FIELD);

        while (true) {
          const queryWithLimit = `${query} limit ${limit} offset ${offset}`;
          const resp = await kintone.api(kintone.api.url('/k/v1/records', true), 'GET', {
            app: kintone.app.getId(),
            query: queryWithLimit,
            fields: fields
          });
          records = records.concat(resp.records);
          if (resp.records.length < limit) break;
          offset += limit;
          updateProgress(records.length, '?', 'レコードリスト取得中');
        }
        return records;
      };

      const records = await fetchAll();

      if (records.length === 0) {
        Swal.fire('情報', '対象レコードが0件です。', 'info');
        return;
      }

      let zipName = '';
      if (CONF.BULK_NAME_TYPE === 'default') {
        try {
          const appInfo = await kintone.api(kintone.api.url('/k/v1/app', true), 'GET', {id: kintone.app.getId()});
          const appName = appInfo.name;
          zipName = `${getTodayDateString()}_${sanitizeFileName(appName)}.zip`;
        } catch(e) {
          const appName = document.title.split(' - ')[0] || 'app';
          zipName = `${getTodayDateString()}_${sanitizeFileName(appName)}.zip`;
        }
      } else {
        const dateStr = CONF.BULK_NAME_ADD_DATE ? getTodayDateString() + '_' : '';
        const customText = CONF.BULK_NAME_TEXT || 'download';
        zipName = `${dateStr}${sanitizeFileName(customText)}.zip`;
      }

      await processDownload(records, targetFields, zipName, {
        isBulk: true,
        folderType: CONF.BULK_FOLDER_TYPE,
        folderField: CONF.BULK_FOLDER_FIELD
      });
    });
  }

  async function handleIndividualDownload(event) {
    const subdomain = location.host.split('.')[0];
    // ★ライセンス認証: 成功時のみ実行
    checkPaidLicense(subdomain, async function() {
      
      const record = event.record;
      let targetFields = [];
      if (CONF.INDIV_TARGET_TYPE === 'fixed') {
        targetFields = CONF.INDIV_TARGET_FIELDS;
      } else {
        const allFields = await getAttachmentFields();
        targetFields = await showFieldSelectModal(allFields);
        if (!targetFields) return;
      }

      if (targetFields.length === 0) {
        Swal.fire('エラー', '対象フィールドが設定されていません。', 'error');
        return;
      }

      let zipName = '';
      if (CONF.INDIV_NAME_TYPE === 'default') {
        zipName = `${getTodayDateString()}_${record['$id'].value}.zip`;
      } else {
        const dateStr = CONF.INDIV_NAME_ADD_DATE ? getTodayDateString() + '_' : '';
        let namePart = 'record';
        if (CONF.INDIV_NAME_FIELD && record[CONF.INDIV_NAME_FIELD]) {
          namePart = record[CONF.INDIV_NAME_FIELD].value || 'record';
        }
        zipName = `${dateStr}${sanitizeFileName(namePart)}.zip`;
      }

      await processDownload([record], targetFields, zipName, {
        isBulk: false,
        folderType: CONF.INDIV_FOLDER_TYPE
      });
    });
  }

  kintone.events.on('app.record.index.show', (event) => {
    if (CONF.MODE === 'individual') return;
    if (document.getElementById('plugin-bulk-download-btn')) return;

    const headerSpace = kintone.app.getHeaderMenuSpaceElement();
    const btn = document.createElement('button');
    btn.id = 'plugin-bulk-download-btn';
    btn.className = 'kintoneplugin-button-normal autonum-header-button';
    btn.innerHTML = '<i class="fa fa-download"></i> 添付一括DL';
    btn.style.marginLeft = '10px';
    btn.textContent = '添付ファイル一括DL';

    btn.onclick = handleBulkDownload;
    headerSpace.appendChild(btn);

    return event;
  });

  kintone.events.on('app.record.detail.show', (event) => {
    if (CONF.MODE === 'bulk') return;
    if (document.getElementById('plugin-indiv-download-btn')) return;

    const headerSpace = kintone.app.record.getHeaderMenuSpaceElement();
    const btn = document.createElement('button');
    btn.id = 'plugin-indiv-download-btn';
    btn.className = 'kintoneplugin-button-normal'; 
    btn.textContent = '添付ファイルをDL';
    btn.style.marginLeft = '10px';

    btn.onclick = () => handleIndividualDownload(event);
    headerSpace.appendChild(btn);

    return event;
  });

})(jQuery, kintone.$PLUGIN_ID);